// Programs for reversing bits.
// Max line length is 57, to fit in hacker.book.
#include <stdio.h>
#include <stdlib.h>     // To define "exit", req'd by XLC.

// Rotate shift left.
unsigned shlr(unsigned x, int n) {
   return (x << n) | (x >> (32 - n));
}

// Reversing bits in a word, basic interchange scheme.
unsigned rev1(unsigned x) {
   x = (x & 0x55555555) <<  1 | (x & 0xAAAAAAAA) >>  1;
   x = (x & 0x33333333) <<  2 | (x & 0xCCCCCCCC) >>  2;
   x = (x & 0x0F0F0F0F) <<  4 | (x & 0xF0F0F0F0) >>  4;
   x = (x & 0x00FF00FF) <<  8 | (x & 0xFF00FF00) >>  8;
   x = (x & 0x0000FFFF) << 16 | (x & 0xFFFF0000) >> 16;
   return x;
}

// Reversing bits in a word, refined basic scheme.
// 24 operations + 6 loads of constants = 30 insns.
// This is Figure 7-1 in HD.
unsigned rev2(unsigned x) {
   x = (x & 0x55555555) <<  1 | (x >>  1) & 0x55555555;
   x = (x & 0x33333333) <<  2 | (x >>  2) & 0x33333333;
   x = (x & 0x0F0F0F0F) <<  4 | (x >>  4) & 0x0F0F0F0F;
   x = (x << 24) | ((x & 0xFF00) << 8) |
       ((x >> 8) & 0xFF00) | (x >> 24);
   return x;
}

// Reversing bits in a word, similar to above but saves shifts.
// Rotate shift required.
// By Vesa Karvonen, found in The Assembly Gems Page.
// 17 operations + 8 to load constants = 25 insns.
// Uses x & ^m = (x & m) ^ x to change the following:
// x = shlr(x & 0x00FF00FF, 16) | x & ~0x00FF00FF;
// to the first executable line of the function.
unsigned rev2a(unsigned x) {
   unsigned t;
   t = x & 0x00FF00FF; x = shlr(t, 16) | t ^ x;
   t = x & 0x0F0F0F0F; x = shlr(t,  8) | t ^ x;
   t = x & 0x33333333; x = shlr(t,  4) | t ^ x;
   t = x & 0x55555555; x = shlr(t,  2) | t ^ x;
   x = shlr(x, 1);
   return x;
}

// Reversing the rightmost 6 bits in a word.
unsigned rev3(unsigned x) {
   return (x*0x00082082 & 0x01122408) % 255;
}

// Reversing the rightmost 6 bits in a word.
unsigned rev4(unsigned x) {
   unsigned t;
   t = x*0x00082082 & 0x01122408;
   return (t*0x01010101) >> 24;
}

// Reversing the rightmost 8 bits in a word.
// Problem: can't change divide by 1023 to a 32x32 ==> 32 multiply.
unsigned rev5(unsigned x) {
   unsigned s, t;
   s = x*0x02020202 & 0x84422010;
   t = x*8 & 0x00000420;
   return (s + t) % 1023;
}

// Reversing the rightmost 8 bits in a word.
unsigned rev6(unsigned x) {
   unsigned s, t;
   s = x*0x00020202 & 0x00422010;
   t = x*0x00008008 & 0x00210420;
   return (s + t) % 1023;
}

// Reversing the rightmost 8 bits in a word.
unsigned rev7(unsigned x) {
   unsigned s, t;

   s = x*0x00020202 & 0x01044010;
   t = x*0x00080808 & 0x02088020;
   return (s + t) % 4095;
}

// Reversing the rightmost 8 bits in a word.
// Same as above but made more efficient, 15 Brisc ops.
unsigned rev8(unsigned x) {
   unsigned u, m, s, t;

   u = x*0x00020202;
   m = 0x01044010;
   s = u & m;
   t = (u << 2) & (m << 1);
   return 0x01001001*(s + t) >> 24;
// return (s + t) % 4095;       // Alternative, equivalent to above.
}

// Reversing the rightmost 9 bits in a word.
// Can't change the remainder to a multiply.
unsigned rev9(unsigned x) {
   unsigned s, t;

   s = x*0x01001001 & 0x84108010;
// t = x*0x00040040 & 0x00841080;
   t = ((x*0x01001001) >> 6) & (0x84108010 >> 8);  // Alternative.
   return (s + t) % 1023;
}

// Reversing the rightmost 9 bits in a word.
// Can't change the remainder to a multiply.
// Probably rev9 is a little better --- smaller numbers.
unsigned rev10(unsigned x) {
   unsigned s, t;

   s = x*0x40100401 & 0x84108010;
   t = x*0x04010040 & 0x21042000;
   return (s + t) % 4095;
}

// Same as above but made more efficient.
unsigned rev10a(unsigned x) {
   unsigned u, m, s, t;

   u = x*0x40100401;
   m = 0x84108010;
   s = u & m;
   t = (u << 6) & (m >> 2);
   return (s + t) % 4095;
}

int errors;
void error(int x, int y) {
   errors = errors + 1;
   printf("Error for x = %08x, got %08x\n", x, y);
}

int main() {
   int i, r, n;
   static unsigned test[] = {0,0, 1,0x80000000, 2,0x40000000,
      3,0xC0000000, 4,0x20000000, 5,0xA0000000, 6,0x60000000,
      7,0xE0000000, 8,0x10000000, 9,0x90000000, 10,0x50000000,
      11,0xD0000000, 12,0x30000000, 13,0xB0000000, 14,0x70000000,
      15,0xF0000000, 16,0x08000000, 59,0xDC000000, 60,0x3C000000,
      61,0xBC000000, 62,0x7C000000, 63,0xFC000000, 64,0x02000000,
      125,0xBE000000, 126,0x7E000000, 127,0xFE000000, 128,0x01000000,
      253,0xBF000000, 254,0x7F000000, 255,0xFF000000, 256,0x00800000,
      257,0x80800000,
      509,0xBF800000, 510,0x7F800000, 511,0xFF800000, 512,0x00400000,
      0x55555555,0xAAAAAAAA,
      0x01234567,0xE6A2C480, 0xFFFFFFF1,0x8FFFFFFF, 0xFFFFFFFF,0xFFFFFFFF};

   n = sizeof(test)/4;

   printf("rev1:\n");
   for (i = 0; i < n; i += 2) {
      r = rev1(test[i]);
      if (r != test[i+1]) error(test[i], r);
      r = rev1(test[i+1]);
      if (r != test[i]) error(test[i+1], r);}

   printf("rev2:\n");
   for (i = 0; i < n; i += 2) {
      r = rev2(test[i]);
      if (r != test[i+1]) error(test[i], r);
      r = rev2(test[i+1]);
      if (r != test[i]) error(test[i+1], r);}

   printf("rev2a:\n");
   for (i = 0; i < n; i += 2) {
      r = rev2a(test[i]);
      if (r != test[i+1]) error(test[i], r);
      r = rev2a(test[i+1]);
      if (r != test[i]) error(test[i+1], r);}

   printf("rev3:\n");
   for (i = 0; i < n; i += 2) {
      if (test[i] > 63) break;
      r = rev3(test[i]);
      if (r != test[i+1] >> 26) error(test[i], r);
      r = rev3(test[i+1] >> 26);
      if (r != test[i]) error(test[i+1], r);}

   printf("rev4:\n");
   for (i = 0; i < n; i += 2) {
      if (test[i] > 63) break;
      r = rev4(test[i]);
      if (r != test[i+1] >> 26) error(test[i], r);
      r = rev4(test[i+1] >> 26);
      if (r != test[i]) error(test[i+1], r);}

   printf("rev5:\n");
   for (i = 0; i < n; i += 2) {
      if (test[i] > 255) break;
      r = rev5(test[i]);
      if (r != test[i+1] >> 24) error(test[i], r);
      r = rev5(test[i+1] >> 24);
      if (r != test[i]) error(test[i+1], r);}

   printf("rev6:\n");
   for (i = 0; i < n; i += 2) {
      if (test[i] > 255) break;
      r = rev6(test[i]);
      if (r != test[i+1] >> 24) error(test[i], r);
      r = rev6(test[i+1] >> 24);
      if (r != test[i]) error(test[i+1], r);}

   printf("rev7:\n");
   for (i = 0; i < n; i += 2) {
      if (test[i] > 255) break;
      r = rev7(test[i]);
      if (r != test[i+1] >> 24) error(test[i], r);
      r = rev7(test[i+1] >> 24);
      if (r != test[i]) error(test[i+1], r);}

   printf("rev8:\n");
   for (i = 0; i < n; i += 2) {
      if (test[i] > 255) break;
      r = rev8(test[i]);
      if (r != test[i+1] >> 24) error(test[i], r);
      r = rev8(test[i+1] >> 24);
      if (r != test[i]) error(test[i+1], r);}

   printf("rev9:\n");
   for (i = 0; i < n; i += 2) {
      if (test[i] > 511) break;
      r = rev9(test[i]);
      if (r != test[i+1] >> 23) error(test[i], r);
      r = rev9(test[i+1] >> 23);
      if (r != test[i]) error(test[i+1], r);}

   printf("rev10:\n");
   for (i = 0; i < n; i += 2) {
      if (test[i] > 511) break;
      r = rev10(test[i]);
      if (r != test[i+1] >> 23) error(test[i], r);
      r = rev10(test[i+1] >> 23);
      if (r != test[i]) error(test[i+1], r);}

   printf("rev10a:\n");
   for (i = 0; i < n; i += 2) {
      if (test[i] > 511) break;
      r = rev10a(test[i]);
      if (r != test[i+1] >> 23) error(test[i], r);
      r = rev10a(test[i+1] >> 23);
      if (r != test[i]) error(test[i+1], r);}

   if (errors == 0)
      printf("Passed all %d cases.\n", sizeof(test)/8);
}
